/*
 * Copyright 2009-2011 Florent Autrusseau.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <math.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include "SDL.h"
#include "SDL_image.h"

#include <it/types.h>
#include <it/fourier.h>
#include <it/io.h>
#include <it/mat.h>
#include <it/parser.h>
#include <it/filter.h>
#include <it/transform.h>
#include <it/wavelet.h>
#include <it/transform2D.h>
#include <it/wavelet2D.h>
#include <it/separable2D.h>
#include <it/random.h>
#include <it/vec.h>
#include <it/cplx.h>


#define FALSE	0
#define TRUE	1

#define SaveData 0
#define UseGamma	0

#define R 0xff
#define G 0xff
#define B 0xff

#if SDL_BYTEORDER == SDL_BIG_ENDIAN
Uint32 rmask = 0xff000000;
Uint32 gmask = 0x00ff0000;
Uint32 bmask = 0x0000ff00;
Uint32 amask = 0x000000ff;
#else
Uint32 rmask = 0x000000ff;
Uint32 gmask = 0x0000ff00;
Uint32 bmask = 0x00ff0000;
Uint32 amask = 0xff000000;
#endif

//SDL_Surface *screen; 

struct tm Today;


/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
/*
 mat gauss_filtering (mat m, int size, double sigma)
 {
 int i, j, k, l;
 int  h = mat_height (m);
 int  w = mat_width (m);
 double sum=0.0;
 mat gauss, filt;
 mat gaussFiltered;
 
 gauss = mat_new(size, size);
 filt = mat_new(size, size);
 gaussFiltered = mat_new(w, h);
 
 //  printf("\n\t%d\n",(int)(lrint(-(size-1)/2)));
 
 k = l = 0;
 for ( i = (int)(lrint(-(size-1)/2)) ; i <= (int)(lrint(size-1)/2) ; i++)
 {
 for ( j = (int)(lrint(-(size-1)/2)) ; j <= (int)(lrint(size-1)/2) ; j++)
 {
 gauss[k][l] = exp( -( pow(i,2) + pow(j,2) ) / (2*(pow(sigma,2)) ));
 l++;
 }
 l=0; k++;
 }
 
 sum = mat_sum(gauss);
 
 for (i = 0 ; i < size ; i++)
 for (j = 0 ; j < size ; j++)
 filt[i][j] = gauss[i][j] / sum;
 
 //  for (i = 0 ; i < 5 ; i++)
 //    for (j = 0 ; j < 5 ; j++)
 //    	printf("%lf\t",filt[i][j]);
 //    printf("\n");
 
 gaussFiltered = mat_filter_fir (m, gauss, 2, 2);
 
 mat_delete(gauss);
 mat_delete(filt);
 
 return (gaussFiltered);
 }
 */
/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
/*
 mat gl_2_lum (mat m)
 {
 int i, j;
 int  h = mat_height (m);
 int  w = mat_width (m);
 mat Lum;
 
 Lum = mat_new (w,h);
 
 for (i = 0 ; i < h ; i++)	{
 for (j = 0 ; j < w ; j++)	{
 Lum[i][j] = (0.7 + 69.3 * (pow((m[i][j] / 255.0),2)));
 }
 }
 
 return(Lum);
 }
 */
/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
/*
 mat lum_2_gl (mat m)
 {
 int i, j;
 int  h = mat_height (m);
 int  w = mat_width (m);
 mat gl;
 
 gl = mat_new (w,h);
 
 for (i = 0 ; i < h ; i++)	{
 for (j = 0 ; j < w ; j++)	{
 gl[i][j] = (255.0 * (pow(((m[i][j] - 0.7) / 69.3),(0.5))));
 }
 }
 //ImOr2 = (255.0 .* (((CA_Rec - 0.7) ./ 69.3)).^(1/2));
 
 return(gl);
 }
 */
/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
void fmat2D_fwrite (FILE * stream, mat m)
{
    int i,j;
    int  w = mat_width (m);
    int  h = mat_height (m);
    
    
    for (i = 0 ; i < h ; i++)	{
	for (j = 0 ; j < w ; j++)	{
	    fprintf(stream, "%lf\t", m[i][j]);
	}
	fprintf(stream,"\n");
    }	
    //  fclose(stream);
}

/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
void create_noise (mat m_z, int w, int h)
{
    int i,j, mat_size;
    double mu, s = 0, k = 0;
    //  mat m_z;
    
    mat_size = w*h;
    
    //  m_z = mat_new (w,h);
    
    for (i = 0; i < h; i++)
	for (j = 0; j < w; j++)
	    m_z[i][j] = it_randn ();
    
    mu = mat_mean (m_z);
    
    for (i = 0; i < h; i++) {
	for (j = 0; j < w; j++) {
	    s += pow (m_z[i][j] - mu, 3.);
	    k += pow (m_z[i][j] - mu, 4.);
	}
    }
    
    s /= (double) mat_size - 1;
    k /= (double) mat_size - 1;
    
    //  printf ("mu=%f\t sigma^2=%f\t skew=%f\t kurt=%f\n",
    //    mat_mean (m_z), mat_variance (m_z), s, k);
    
    //  printf ("JB=%e\n", (double) (mat_size) / 6. * (s * s +
    //                   ((k - 3) * (k - 3))) / 4.);
    
    //return(m_z);
    
}

/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
void add_dwt_sb_smooth(mat m_markedsb, mat m_wn, mat mt, int level, int orientation, double a, double b, int curve)
{
    int i, j, w, h;
    
    w = mat_width (mt);
    h = mat_height (mt);
    
    mat_copy (m_markedsb, mt);

    for (i = 0; i < h; i++) {
	for (j = 0; j < w; j++) {
	    if (orientation == 1)
	    {
		if (((i >= 0) && (i < (int)( lrint(h / pow(2,level)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    if (curve == 1)
//			m_markedsb[i][j] = mt[i][j] + (a * fabs(mt[i][j]) + b) * m_wn[i][j] * fabs(mt[i][j]);
			m_markedsb[i][j] = mt[i][j] + (a * fabs(mt[i][j]) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 2)
//			m_markedsb[i][j] = mt[i][j] + (a * pow(mt[i][j],2) + b) * m_wn[i][j] * fabs(mt[i][j]);
			m_markedsb[i][j] = mt[i][j] + (a * pow(mt[i][j],2) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 3)
//			m_markedsb[i][j] = mt[i][j] + (a * pow(fabs(mt[i][j]),0.5) + b) * m_wn[i][j] * fabs(mt[i][j]);
			m_markedsb[i][j] = mt[i][j] + (a * pow(fabs(mt[i][j]),0.5) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		}
	    }
	    else if (orientation == 2)
	    {
		if (((i >= (int)( lrint(h / pow(2,level)) )) && (i < (int)( lrint(h / pow(2,level-1)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    if (curve == 1)
			m_markedsb[i][j] = mt[i][j] + (a * fabs(mt[i][j]) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 2)
			m_markedsb[i][j] = mt[i][j] + (a * pow(mt[i][j],2) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 3)
			m_markedsb[i][j] = mt[i][j] + (a * pow(fabs(mt[i][j]),0.5) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		}
	    }
	    else if (orientation == 3)
	    {
		if (((i >= (int)( lrint(h / pow(2,level)) )) && (i < (int)( lrint(h / pow(2,level-1)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    if (curve == 1)
			m_markedsb[i][j] = mt[i][j] + (a * fabs(mt[i][j]) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 2)
			m_markedsb[i][j] = mt[i][j] + (a * pow(mt[i][j],2) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 3)
			m_markedsb[i][j] = mt[i][j] + (a * pow(fabs(mt[i][j]),0.5) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		}
	    }
	    else if (orientation == 4)
	    {
		if (((i >= 0) && (i < (int)( lrint(h / pow(2,level)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    if (curve == 1)
			m_markedsb[i][j] = mt[i][j] + (a * fabs(mt[i][j]) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 2)
			m_markedsb[i][j] = mt[i][j] + (a * pow(mt[i][j],2) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		    else if (curve == 3)
			m_markedsb[i][j] = mt[i][j] + (a * pow(fabs(mt[i][j]),0.5) + b) * m_wn[i][j];// * fabs(mt[i][j]);
		}
	    }
	}
    }
}

/*----------------------------------------------------*/
/*                    function                        */
/*----------------------------------------------------*/
void getRange(mat mt, int level, int orientation)
{
    int i, j, w, h;
    double min1=1000, min2=1000, min3=1000, min4=1000;
    double max1=0, max2=0, max3=0, max4=0;
    double sum=0.0;
    double mean=0.0;
    double stdev=0.0;
    
    w = mat_width (mt);
    h = mat_height (mt);
    
    
    for (i = 0; i < h; i++) {
	for (j = 0; j < w; j++) {
	    if (orientation == 1)
	    {
		if (((i >= 0) && (i < (int)( lrint(w / pow(2,level)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    if (mt[i][j] <= min1)	min1 = mt[i][j];
		    if (mt[i][j] >= max1)	max1 = mt[i][j];
		    sum += mt[i][j];
		}
	    }
	    else if (orientation == 2)
	    {
		if (((i >= (int)( lrint(w / pow(2,level)) )) && (i < (int)( lrint(w / pow(2,level-1)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    if (mt[i][j] <= min2)	min2 = mt[i][j];
		    if (mt[i][j] >= max2)	max2 = mt[i][j];
		    sum += mt[i][j];
		}
	    }
	    else if (orientation == 3)
	    {
		if (((i >= (int)( lrint(w / pow(2,level)) )) && (i < (int)( lrint(w / pow(2,level-1)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    if (mt[i][j] <= min3)	min3 = mt[i][j];
		    if (mt[i][j] >= max3)	max3 = mt[i][j];
		    sum += mt[i][j];
		}
	    }
	    else if (orientation == 4)
	    {
		if (((i >= 0) && (i < (int)( lrint(w / pow(2,level)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    if (mt[i][j] <= min3)	min3 = mt[i][j];
		    if (mt[i][j] >= max4)	max4 = mt[i][j];
		    sum += mt[i][j];
		}
	    }
	}
    }
    
    if (orientation == 1)
	mean = sum / (pow((int)( lrint(w / pow(2,level)) ),2));
    else if (orientation == 2)
	mean = sum / (pow((int)( lrint(w / pow(2,level-1)) ) - (int)( lrint(w / pow(2,level)) ),2));
    else if (orientation == 3)
	mean = sum / (pow((int)( lrint(w / pow(2,level-1)) ) - (int)( lrint(w / pow(2,level)) ),2));
    else if (orientation == 4)
	mean = sum / (pow((int)( lrint(w / pow(2,level)) ),2));
    
    // printf("mean: %lf\n",mean);
    
    sum = 0.0;
    
    for (i = 0; i < h; i++) {
	for (j = 0; j < w; j++) {
	    if (orientation == 1)
	    {
		if (((i >= 0) && (i < (int)( lrint(w / pow(2,level)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    sum += pow((mt[i][j] - mean),2);
		}
	    }
	    else if (orientation == 2)
	    {
		if (((i >= (int)( lrint(w / pow(2,level)) )) && (i < (int)( lrint(w / pow(2,level-1)) ))) && 
		    ((j >= (int)( lrint(w / pow(2,level)) )) && (j < (int)( lrint(w / pow(2,level-1)) ))))	{
		    sum += pow((mt[i][j] - mean),2);
		}
	    }
	    else if (orientation == 3)
	    {
		if (((i >= (int)( lrint(w / pow(2,level)) )) && (i < (int)( lrint(w / pow(2,level-1)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    sum += pow((mt[i][j] - mean),2);
		}
	    }
	    else if (orientation == 4)
	    {
		if (((i >= 0) && (i < (int)( lrint(w / pow(2,level)) ))) && 
		    ((j >= 0) && (j < (int)( lrint(w / pow(2,level)) ))))	{
		    sum += pow((mt[i][j] - mean),2);
		}
	    }
	}
    }
    
    if (orientation == 1)
	stdev = sqrt((1/pow(((int)( lrint(w / pow(2,level)) )),2)) * sum);
    else if (orientation == 2)
	stdev = sqrt((pow((int)( lrint(w / pow(2,level-1)) ) - (int)( lrint(w / pow(2,level)) ),2)) * sum);
    else if (orientation == 3)
	stdev = sqrt((pow((int)( lrint(w / pow(2,level-1)) ) - (int)( lrint(w / pow(2,level)) ),2)) * sum);
    else if (orientation == 4)
	stdev = sqrt((1/pow(((int)( lrint(w / pow(2,level)) )),2)) * sum);
    
    printf("stdev: %lf\n",stdev);

    if (orientation == 1)
	printf("SB range: [%lf, %lf]\n", min1, max1);
    else if (orientation == 2)
	printf("SB range: [%lf, %lf]\n", min2, max2);
    else if (orientation == 3)
	printf("SB range: [%lf, %lf]\n", min3, max3);
    else if (orientation == 4)
	printf("SB range: [%lf, %lf]\n", min4, max4);
    
}


/**************************************************************************/
void DrawPixel(SDL_Surface *screen, int x, int y, Uint8 Red, Uint8 Green, Uint8 Blue) 
{ 
    Uint32 color = SDL_MapRGB(screen->format, Red, Green, Blue);
    switch (screen->format->BytesPerPixel) 
    { 
	case 1: // 8-bpp 
	{ 
	    Uint8 *bufp; 
	    bufp = (Uint8 *)screen->pixels + y*screen->pitch + x; 
	    *bufp = color; 
	} 
	    break; 
	case 2: // 15-bpp or 16-bpp 
	{ 
	    Uint16 *bufp; 
	    bufp = (Uint16 *)screen->pixels + y*screen->pitch/2 + x; 
	    *bufp = color; 
	} 
	    break; 
	case 3: // 24-bpp mode, usually not used 
	{ 
	    Uint8 *bufp; 
	    bufp = (Uint8 *)screen->pixels + y*screen->pitch + x * 3; 
	    if(SDL_BYTEORDER == SDL_LIL_ENDIAN) 
	    { 
		bufp[0] = color; 
		bufp[1] = color >> 8; 
		bufp[2] = color >> 16; 
	    } else { 
		bufp[2] = color; 
		bufp[1] = color >> 8; 
		bufp[0] = color >> 16; 
	    } 
	} 
	    break; 
	case 4: // 32-bpp 
	{ 
	    Uint32 *bufp; 
	    bufp = (Uint32 *)screen->pixels + y*screen->pitch/4 + x; 
	    *bufp = color; 
	} 
	    break; 
    } 
}

void DrawIMG(SDL_Surface *img, SDL_Surface* des, int x, int y)
{
  SDL_Rect dest;
  dest.x = x;
  dest.y = y;
  SDL_BlitSurface(img, NULL, des, &dest);
}

void shuffle(int *list, int n) {

	int i, j, k;

  /* initialize random seed: */
  srand ( time(NULL) );

	for (i = 0; i < n; i++)
		list[i] = i;

	for (i = 0; i < n; i++) {
		j = rand() % n;
		k = list[i];
		list[i] = list[j];
		list[j] = k;
	}
	for (i = 0; i < n; i++)
		list[i] = list[i] +1;
}

/**************************************************************************************************/
int main(int argc, char **argv)
{
    SDL_Surface *screen, *imageO, *imageM;
    SDL_Event event;
    int SavedBMP=0;
    int done=0;
    int a;
    int i,j,k;
    int b;
    int curve;
    
    double acoef = 0.00002, bcoef = 0.00002;
    double tmpr, tmpg, tmpb;
    double *ptr_a, *ptr_b;

    int  nb_levels;
    int level;
    int orientation;
    int *PTRnb_levels;
    int *PTRlevel;
    int *PTRorientation;
    
    int w, h;
    char *name[256];
    char *outputfilename;
    char *markedfilename;
//	time_t rawtime;
//	struct tm * timeinfo;
    int *list;
	int k2=0;    
//    char *nameWithpath[256];
    int NbImages;
    
    it_wavelet2D_t *wavelet2D;
    it_wavelet2D_t *wavelet2Db;
    it_wavelet2D_t *wavelet2DNoise;

    mat mt, gm;
    mat m_z;
    mat m_wn;
    mat m_markedsb;
    mat m_marked;
    mat mgl_marked;
    mat matrixY, matrixCr, matrixCb;

    time_t now;

    //	FILE *matFile=NULL;
    FILE * pFile;
    FILE * pFile2;

    FILE *fptr;


  if (argv[1] == NULL)
  {
//    if ((fptr = fopen("/Users/florent/Desktop/WavJNDStuff/wavjnd/playlist2.txt","r")) == NULL)
    if ((fptr = fopen("playlist2.txt","r")) == NULL)
    { printf("problem while opening playlist2.txt\n");exit(1);}
  }
  else
  {
    if ((fptr = fopen(argv[1],"r")) == NULL)
    { printf("problem while opening argv[1]\n");exit(1);}
  }
    fscanf(fptr,"%d\n",&curve);
    printf("curve: %d (1: linear, 2: ^2, 3: sqrt\n",curve);

    fscanf(fptr,"%d\n",&NbImages);
    printf("%d  images\n",NbImages);

    for (i=1;i<NbImages+2;i++)
	name[i] = (char *) calloc (256,sizeof(char));

	outputfilename = (char *) calloc (256,sizeof(char));

//    for (i=0;i<NbImages;i++)
//	nameWithpath[i] = (char *) calloc (256,sizeof(char));

    PTRnb_levels = (int *) calloc (50,sizeof(int));
    PTRlevel = (int *) calloc (50,sizeof(int));
    PTRorientation = (int *) calloc (50,sizeof(int));

    for (i = 1 ; i < NbImages+1 ; i++)  {
	a = fscanf(fptr,"%s\n",name[i]);
	printf("%s\n",name[i]);
	a = fscanf(fptr,"%d\n",&PTRnb_levels[i]);
	printf("%d\n",PTRnb_levels[i]);
	a = fscanf(fptr,"%d\n",&PTRlevel[i]);
	printf("%d\n",PTRlevel[i]);
	a = fscanf(fptr,"%d\n",&PTRorientation[i]);
	printf("%d\n",PTRorientation[i]);
    }
    fclose(fptr);
    
//	time ( &rawtime );
//	timeinfo = localtime ( &rawtime );
//	printf ( "The current date/time is: %s", asctime (timeinfo) );
//	time ( &rawtime );
//    sprintf(outputfilename,"output_%s.log",ctime (&rawtime));

	time(&now);
	Today = *localtime(&now);

//	printf("%4.4d/%2.2d/%2.2d\n",Today.tm_year + 1900, Today.tm_mon + 1, Today.tm_mday);
	sprintf(outputfilename,"output_%4.4d%2.2d%2.2d_%2.2d_%2.2d_%2.2d.log",
				Today.tm_year + 1900, Today.tm_mon + 1, Today.tm_mday, Today.tm_hour, Today.tm_min, Today.tm_sec);

//    pFile = fopen ("/Users/florent/Desktop/WavJNDStuff/wavjnd/output.log","w");
//    pFile = fopen ("output.log","w");
    pFile = fopen (outputfilename,"w");

    fprintf(pFile,"curve: %d (1: linear, 2: ^2, 3: sqrt)\n",curve);
    fprintf(pFile,"time: %4.4d%2.2d%2.2d_%2.2d_%2.2d_%2.2d\n",
    		Today.tm_year + 1900, Today.tm_mon + 1, Today.tm_mday, Today.tm_hour, Today.tm_min, Today.tm_sec);
    
    
    /* startup SDL */
    if(SDL_Init(SDL_INIT_VIDEO)==-1)	{
	printf("SDL_Init: %s\n", SDL_GetError());
	return 1;
    }
    
    
    const SDL_VideoInfo* vidinfo = SDL_GetVideoInfo();
    
    /* open the screen for displaying the image */
    /* try to match the image size and depth    */
    screen=SDL_SetVideoMode(vidinfo->current_w, vidinfo->current_h, 24, SDL_FULLSCREEN); //SDL_ANYFORMAT);
//    screen=SDL_SetVideoMode(vidinfo->current_w, vidinfo->current_h, 24, SDL_ANYFORMAT);
    if(!screen)	{
	printf("SDL_SetVideoMode: %s\n", SDL_GetError());
	SDL_Quit();
	return 1;
    }

	/* Fill the background with Gray color (128,128,128) */
	SDL_FillRect( screen, NULL, SDL_MapRGBA(screen->format, 128, 128, 128, 255) );
	//SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 128, 128, 128));

//    imageM = SDL_CreateRGBSurface(SDL_SWSURFACE, 512, 512, 24, 0,0,0,0);
//    if(imageM == NULL) {
//	fprintf(stderr, "CreateRGBSurface failed: %s\n", SDL_GetError());
//	exit(1);
//    }

    markedfilename = (char *) calloc (256,sizeof(char));

    imageO = SDL_CreateRGBSurface(SDL_SWSURFACE | SDL_SRCALPHA, vidinfo->current_w, vidinfo->current_h, 24, 0,0,0,0);

	// 'SHUFFLING' the playlist...
	list = (int *) calloc (NbImages+2,sizeof(int));

	shuffle(list, NbImages);

	printf("\nShuffled list of input images:\n");
	for (i=0;i<NbImages;i++)
		printf("%d\n",list[i]);

	// End of the 'SHUFFLING' algo.

	ptr_a = (double *) calloc (NbImages+2,sizeof(double));
	ptr_b = (double *) calloc (NbImages+2,sizeof(double));

	k2=0;

    /********      LOOP ON ALL THE PLAYLIST'S IMAGES      ********/
    for (k=0;k<NbImages+2;k++)
    {
    if (k == 0)
    {
	imageO=IMG_Load("intro.bmp");
	if(!imageO)	{
		/* image failed to load */
		printf("IMG_Load (intro): %s\n", IMG_GetError());
		SDL_Quit();
		return 1;
	}
    }
	if (k == NbImages+1)
	{
//	    imageO=IMG_Load("/Users/florent/Desktop/WavJNDStuff/wavjnd/thanks.bmp");
	    imageO=IMG_Load("thanks.bmp");
	    if(!imageO)	{
		/* image failed to load */
		printf("IMG_Load (thanks): %s\n", IMG_GetError());
		SDL_Quit();
		return 1;
	    }
	}
	else if ((k != 0) && (k != NbImages+1)) 
	{
		k2=list[k-1];
//	    printf("%s\n",nameWithpath[k]);
	    /* load the image */      
//	   sprintf(nameWithpath[k],"/Users/florent/Desktop/WavJNDStuff/wavjnd/%s",name[k]);	    
//	   imageO=IMG_Load("/Users/florent/Desktop/WavJNDStuff/wavjnd/barb512.bmp");
	   imageO=IMG_Load(name[k2]);
//	   imageO=IMG_Load(nameWithpath[k]);
	    if(!imageO)		{
		/* image failed to load */
		printf("IMG_Load: %s\n", IMG_GetError());
		printf("k2 : %d, name : %s\n", k2, name[k2]);
		SDL_Quit();
		return 1;
	    }
	}
	
//	SDL_FillRect( screen, NULL, SDL_MapRGBA(screen->format, 128, 128, 128, 255) );
//	SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 128, 128, 128));

	nb_levels = PTRnb_levels[k2];
	level = PTRlevel[k2];
	orientation = PTRorientation[k2];
	
	/* print some info about the image */
	if ((k != 0) && (k != NbImages+1))
	    printf("loaded %s: %dx%d %dbpp, marking SB (%d,%d)\n", name[k2],
		   imageO->w, imageO->h, imageO->format->BitsPerPixel, level, orientation);

	/** FADE OUT **/
    imageM = SDL_CreateRGBSurface(SDL_SWSURFACE | SDL_SRCALPHA, imageO->w, imageO->h, 24, 0,0,0,0);
	/** END FADE OUT **/

//    imageM = SDL_CreateRGBSurface(SDL_SWSURFACE, imageO->w, imageO->h, 24, 0,0,0,0);
    if(imageM == NULL) {
	fprintf(stderr, "CreateRGBSurface failed: %s\n", SDL_GetError());
	exit(1);
    }
	
	
	//		const SDL_VideoInfo* vidinfo = SDL_GetVideoInfo();
	//
	//		/* open the screen for displaying the image */
	//		/* try to match the image size and depth    */
	//		screen=SDL_SetVideoMode(vidinfo->current_w, vidinfo->current_h, imageO->format->BitsPerPixel, SDL_FULLSCREEN);
	//		if(!screen)	{
	//			printf("SDL_SetVideoMode: %s\n", SDL_GetError());
	//			SDL_FreeSurface(imageO);
	//			SDL_Quit();
	//			return 1;
	//		}
	
	/* set the window title to the filename */
	//	SDL_WM_SetCaption(argv[1],argv[1]);

	matrixY = mat_new (imageO->h, imageO->w);
	matrixCr = mat_new (imageO->h, imageO->w);
	matrixCb = mat_new (imageO->h, imageO->w);

	mt = mat_new (imageO->h, imageO->w);
	gm = mat_new (imageO->h, imageO->w);
	m_z = mat_new (imageO->h, imageO->w);
	m_wn = mat_new (imageO->h, imageO->w);
	m_markedsb = mat_new (imageO->h, imageO->w);
	mgl_marked = mat_new (imageO->h, imageO->w);

	//-------------------------------------------
	if( SDL_MUSTLOCK(imageO) )
	    b = SDL_LockSurface(imageO);
	
	int bpp = imageO->format->BytesPerPixel;
	
	for (i=0;i<imageO->h;i++)	{
	    for (j=0;j<imageO->w;j++)	{
		Uint8 *p = (Uint8 *)imageO->pixels + i * imageO->pitch + j * bpp;
		tmpr = (double) p[2];
		tmpg = (double) p[1];
		tmpb = (double) p[0];
		matrixY[i][j] = (0.257 * tmpr) + (0.504 * tmpg) + (0.098 * tmpb) + 16;
		matrixCr[i][j] = (0.439 * tmpr) - (0.368 * tmpg) - (0.071 * tmpb) + 128;
		matrixCb[i][j] = -(0.148 * tmpr) - (0.291 * tmpg) + (0.439 * tmpb) + 128;
	    }
	}
	
	if( SDL_MUSTLOCK(imageO) )
	    SDL_UnlockSurface(imageO);


	if (UseGamma == 1)	{
		for (i = 0 ; i <imageO->h ; i++)
		    for (j = 0 ; j < imageO->w ; j++)
			gm[i][j] = (0.26 + 213 * (pow((matrixY[i][j] / 255.0),1.85)));
	}
	else	{
//		printf("\nNOT using Gamma function of the screen...\n");
		mat_copy(gm, matrixY);
	}
	
	/* create a 2D wavelet object */
	wavelet2D = it_wavelet2D_new (it_wavelet_lifting_97, nb_levels);
	
	/* Transform the image */
	mt = it_wavelet2D_transform (wavelet2D, gm);
	mat_delete (gm);

	getRange(mt, level, orientation);
	
	//  if (SaveData)	{
	//  matFile = fopen("dwt_im_or.tim","w");
	//  fmat2D_fwrite(matFile,mt);
	//  fclose(matFile);	}
	
	w = mat_width (mt);
	h = mat_height (mt);
	
	it_randomize();
	create_noise (m_z, w, h);
	
	//  if (SaveData)	{
	//  matFile = fopen("noise.tim","w");
	//  fmat2D_fwrite(matFile,m_z);
	//  fclose(matFile);	}
	
	wavelet2DNoise = it_wavelet2D_new (it_wavelet_lifting_97, nb_levels);
	
	/* Transform the noise to wavelet_noise (m_wn) */
	m_wn = it_wavelet2D_transform (wavelet2DNoise, m_z);
	
	it_delete(wavelet2DNoise);
	mat_delete(m_z);
	
	
	//  if (SaveData)	{
	//  matFile = fopen("dwt_noise.tim","w");
	//  fmat2D_fwrite(matFile,m_wn);
	//  fclose(matFile);	}
	
	// option: filter the watermark before embedding.
	
	//  if (orientation == 4)
	//    m_wn = gauss_filtering (m_wn, FiltSize, sigma);
	
	//    -----------------------
	//    | 2,4 | 2,1 |          |
	//    |     |     |   1,1    |
	//    |-----------|          |
	//    | 2,3 | 2,2 |          |
	//    |     |     |          |
	//    |-----------|----------|
	//    |           |          |
	//    |           |          |
	//    |    1,3    |  1,2     |
	//    |           |          |
	//    |           |          |
	//    -----------------------
	
	
	//  m_markedsb = add_dwt_sb(m_wn, mt, alpha1);
	//		mat_copy (m_markedsb, mt);
	
	
	//		imageM = SDL_CreateRGBSurface(imageO->flags, imageO->w, imageO->h, imageO->format->BitsPerPixel, 0,0,0,0);
	//		if(imageM == NULL) {
	//        	fprintf(stderr, "CreateRGBSurface failed: %s\n", SDL_GetError());
	//        	exit(1);
	//    	}
	
//	SDL_FreeSurface(imageO);

	if( SDL_MUSTLOCK(imageM) )
	    b = SDL_LockSurface(imageM);
	
	for (i=0;i<imageM->h;i++)	{
	    for (j=0;j<imageM->w;j++)	{
		tmpb = 1.164 * (matrixY[i][j] - 16) + 2.018 * (matrixCb[i][j] - 128);
		tmpg = 1.164 * (matrixY[i][j] - 16) - 0.813 * (matrixCr[i][j] - 128) - 0.391 * (matrixCb[i][j] - 128);
		tmpr = 1.164 * (matrixY[i][j] - 16) + 1.596 * (matrixCr[i][j] - 128);
		DrawPixel(imageM, j, i,  (Uint8) (tmpr), (Uint8) (tmpg), (Uint8) (tmpb));
	    }
	}
	
	if( SDL_MUSTLOCK(imageM) )
	    SDL_UnlockSurface(imageM);

	//-------------------------------------------
	
	/* do the initial image display */
	SDL_Rect rcDest = { ((int) (vidinfo->current_w/2) - (int)(imageM->w/2)), 
	((int) (vidinfo->current_h/2) - (int)(imageM->h/2)), imageM->w, imageM->h }; // ... , 512, 512;
	SDL_BlitSurface(imageM,0,screen,&rcDest);
	SDL_Flip(screen);


	/* key repeat so we don't have to keep pressing over and over... */
	SDL_EnableKeyRepeat(1,1);

	acoef = 0.00002;
	bcoef = 0.00002;
	done = 0;
	int inc=0;
	double step = 2.0;

/** FADE OUT **/
	// Timer variable
//	int start = SDL_GetTicks();	
//	printf("start: %d\n",start);
	// Start the screen as Opaque(255)
//	int AlphaValue = 255;
/** END FADE OUT **/

	/* the event loop, redraws if needed, quits on keypress or quit event */
	while(!done && SDL_WaitEvent(&event)!=-1)
	{
	    switch(event.type)
	    {
		case SDL_KEYDOWN:
		{
		    int handled=0;
		    switch(event.key.keysym.sym)
		    {
			case SDLK_ESCAPE:
//				SDL_FreeSurface(imageO);
//				SDL_FreeSurface(imageM);
				SDL_FreeSurface(screen);
				SDL_Quit();

				sprintf(outputfilename,"outputRand_%4.4d%2.2d%2.2d_%2.2d_%2.2d_%2.2d.log",
				Today.tm_year + 1900, Today.tm_mon + 1, Today.tm_mday, Today.tm_hour, Today.tm_min, Today.tm_sec);

			    pFile2 = fopen (outputfilename,"w");
				for (k=1;k<NbImages+1;k++)	{
					fprintf(pFile2,"im: %s\t%12.12lf\t%12.12lf\n",name[k],ptr_a[k],ptr_b[k]);
				}

			    exit(0);
			    break;
			case SDLK_o:
			    acoef = 0.000000002;
			    bcoef = 0.000000002;
			    handled = 1;
			    break;
			case SDLK_1:
			    step = 1.0;
			    break;
			case SDLK_2:
			    step = 2.0;
			    break;
			case SDLK_3:
			    step = 3.0;
			    break;
			case SDLK_4:
			    step = 4.0;
			    break;
			case SDLK_5:
			    step = 5.0;
			    break;
			case SDLK_6:
			    step = 6.0;
			    break;
			case SDLK_7:
			    step = 7.0;
			    break;
			case SDLK_8:
			    step = 8.0;
			    break;
			case SDLK_9:
			    step = 9.0;
			    break;
			case SDLK_UP:
			    acoef *= step;
			    handled=1;
			    break;
			case SDLK_DOWN:
			    acoef /= step;
			    handled=1;
			    break;
			case SDLK_RIGHT:
				if (inc == 1)	{
			    bcoef *=step;
			    handled=1;	}
			    break;
			case SDLK_LEFT:
				if (inc == 1)	{
			    bcoef /= step;
			    handled=1;	}
			    break;
			case SDLK_SPACE:
//			    done=1;
//			    handled=1;
			    inc=1;
			    break;
			case SDLK_RETURN:
				if (k == 0)
					{ done = 1; }
				else if (inc == 1)	{
					if ((acoef != 0.000000002) && (bcoef != 0.000000002))	{
					    done=1;
					    handled=1;
					}
				}
/*
			case SDLK_SPACE:
//			    done=1;
//				if (inc < 2)	{
			    inc++;
			    handled=1;
//				}
				if (inc == 2)	{
					if ((acoef != 0.000000002) && (bcoef != 0.000000002))	{
					inc=0;
					handled=1;
					done=1;
					}
				}
			    break;
			case SDLK_RETURN:
				if (inc == 1)	{
					if ((acoef != 0.000000002) && (bcoef != 0.000000002))	{
					    done=1;
					    handled=1;
					}
				}
*/
			    break;
			default:
			    break;
		    }
		    if(handled)
		    {
//			m_marked = mat_new (imageO->h, imageO->w);

			add_dwt_sb_smooth(m_markedsb, m_wn, mt, level, orientation, acoef, bcoef, curve);

			/* create a 2D wavelet object */
			wavelet2Db = it_wavelet2D_new (it_wavelet_lifting_97, nb_levels);

			/* Transform the noise to wavelet_noise (m_wn) */
			m_marked = it_wavelet2D_itransform (wavelet2Db, m_markedsb);  			

			it_delete(wavelet2Db);
//			mgl_marked = mat_new (imageO->h, imageO->w);

			if (UseGamma == 1)	{
				for (i = 0 ; i < imageM->h ; i++)
				    for (j = 0 ; j < imageM->w ; j++)
						mgl_marked[i][j] = (255.0 * (pow(((m_marked[i][j] - 0.26) / 213),(1/1.85))));
			}
			else {
//				printf("\nNOT using Gamma function of the screen...\n");
				mat_copy(mgl_marked, m_marked);
			}

			mat_delete(m_marked);

			if( SDL_MUSTLOCK(imageM) )
			    b = SDL_LockSurface(imageM);

			for (i=0;i<imageM->h;i++)	{
			    for (j=0;j<imageM->w;j++)	{
				tmpb = 1.164 * (mgl_marked[i][j] - 16) + 2.018 * (matrixCb[i][j] - 128);
				if (tmpb < 0)		tmpb = 0;
				else if (tmpb > 255)		tmpb = 255;
				tmpg = 1.164 * (mgl_marked[i][j] - 16) - 0.813 * (matrixCr[i][j] - 128) - 
				0.391 * (matrixCb[i][j] - 128);
				if (tmpg < 0)		tmpg = 0;
				else if (tmpg > 255)		tmpg = 255;
				tmpr = 1.164 * (mgl_marked[i][j] - 16) + 1.596 * (matrixCr[i][j] - 128);
				if (tmpr < 0)		tmpr = 0;
				else if (tmpr > 255)		tmpr = 255;
				DrawPixel(imageM, j, i,  (Uint8) (tmpr), (Uint8) (tmpg), (Uint8) (tmpb));
			    }
			}
//			mat_delete(mgl_marked);

			if( SDL_MUSTLOCK(imageM) )
			    SDL_UnlockSurface(imageM);

			SDL_BlitSurface(imageM,0,screen,&rcDest);
			SDL_Flip(screen);

			if (done == 1)
			{
				if ((k != 0) && (k != NbImages+1))	{
					sprintf(markedfilename,"marked_sb.%d.%d_%s",PTRlevel[k2], PTRorientation[k2], name[k2]);
					SavedBMP = SDL_SaveBMP(imageM, markedfilename);
				}
				/** FADE OUT **/
				int AlphaValue = 255;
				SDL_SetAlpha( imageM, SDL_SRCALPHA, AlphaValue);
//				printf("%d\t%d\t\t",SDL_GetTicks(), start);
//				while(( SDL_GetTicks() - start > 100 ) && (AlphaValue > 0))
				while(AlphaValue > 0)
				{
//					printf("%d\t%d\n",SDL_GetTicks(), start);
				
					// Decrease the alpha value by this much
					AlphaValue-=50;

//					if( SDL_MUSTLOCK(imageM) )
//					    b = SDL_LockSurface(imageM);

//					SDL_SetAlpha( imageM, SDL_SRCALPHA, AlphaValue);

					// Clear the screen
					SDL_FillRect( imageM, NULL, SDL_MapRGBA(screen->format, 128, 128, 128, AlphaValue) );
//					SDL_FillRect( screen, NULL, SDL_MapRGBA(screen->format, 128, 128, 128, AlphaValue) );

					// Draw the constructed surface to the primary surface now
//					DrawIMG( imageM, screen, ((int) (vidinfo->current_w/2) - (int)(imageM->w/2)), 
//							((int) (vidinfo->current_h/2) - (int)(imageM->h/2)) );
//				if( SDL_MUSTLOCK(imageM) )
//				    SDL_UnlockSurface(imageM);

					SDL_BlitSurface(imageM,0,screen,&rcDest);
					SDL_Flip( screen );

				}
				/** END FADE OUT **/
			}

			break;
		    }
		}
		    //				case SDL_QUIT:
		    //					/* quit events, exit the event loop */
		    //					done=1;
		    //					break;
		    //				default:
		    //					break;
	    }
	} // End 'while'

//	sprintf(markedfilename,"marked_sb.%d.%d_%s",PTRlevel[k], PTRorientation[k], name[k]);
//	SavedBMP = SDL_SaveBMP(imageM, markedfilename);

	if ((k != 0) && (k != NbImages+1)) {
		ptr_a[k2] = acoef;
		ptr_b[k2] = bcoef;
	    printf("acoef : %12.12lf\nbcoef : %12.12lf\n",acoef, bcoef);
	    fprintf(pFile,"im: %s, wavSB: [%d,%d]\n",name[k2], level, orientation);
	    fprintf(pFile,"acoef : %12.12lf\nbcoef : %12.12lf\n\n",acoef, bcoef);
	}
	
	it_delete(wavelet2D);
	
	mat_delete(mgl_marked);
//	mat_delete(m_marked);
	mat_delete(m_markedsb);
	mat_delete(mt);
	mat_delete(m_wn);
	mat_delete(matrixY);
	mat_delete(matrixCr);
	mat_delete(matrixCb);

//	SDL_FreeSurface(imageO);

	/* free the loaded image */
	//	SDL_FreeSurface(screen);
	SDL_FreeSurface(imageM);

    } // End 'for (k)

	sprintf(outputfilename,"outputRand_%4.4d%2.2d%2.2d_%2.2d_%2.2d_%2.2d.log",
				Today.tm_year + 1900, Today.tm_mon + 1, Today.tm_mday, Today.tm_hour, Today.tm_min, Today.tm_sec);

    pFile2 = fopen (outputfilename,"w");
	for (k=1;k<NbImages+1;k++)	{
		fprintf(pFile2,"im: %s\t%12.12lf\t%12.12lf\n",name[k],ptr_a[k],ptr_b[k]);
	}	

    SDL_FreeSurface(screen);
//    SDL_FreeSurface(imageM);

    for (i=1;i<NbImages+2;i++)	{
	free(name[i]);
//	free(nameWithpath[i]);
    }
    
    /* shutdown SDL */
    //SDL_Quit();

    free(list);
	free(ptr_a);
	free(ptr_b);
	
    fclose (pFile);
    fclose (pFile2);
    
    free(PTRnb_levels);
    free(PTRlevel);
    free(PTRorientation);
    
    return 0;
}
